implementation module monitor


import StdEnv, StdIO


::	Count
	=	{	oks			:: Int				// nr of OK keys
		,	bads		:: Int				// nr of correction keys
		}
::	Monitor
	=	{	count		:: Count			// current  registration of keys
		,	counts		:: [Count]			// previous registration of keys
		,	time		:: Int				// current tracking time since start (in seconds)
		,	tracking	:: Bool				// monitor is currently tracking
		}
::	MonitorMessage							// The message type of the monitor process
	=	BeginSession						// Begin a typist session
	|	KeyHit Char							// Register key stroke
	|	EndSession							// End a typist session


openMonitor :: ItemPos (RId MonitorMessage) (PSt .l .p) -> PSt .l .p
openMonitor pos monitorId ps
	#	(font,   ps)= accPIO (accScreenPicture openDialogFont) ps
	#	(metrics,ps)= accPIO (accScreenPicture (getFontMetrics font)) ps
	#	(ids,    ps)= accPIO (openIds 2) ps
	#	ps			= openProcesses (ProcessGroup 0 (monitorProcess pos font metrics monitorId ids)) ps
	=	ps

monitorProcess :: ItemPos Font FontMetrics (RId MonitorMessage) [Id] -> SDIProcess (Window NilLS) .p
monitorProcess pos font metrics monitorId ids=:[wId,tId]
	=	SDIProcess initLocal undef window [initIO] [ProcessShareGUI]
where
	initcount	= {oks=0,bads=0}
	initcounts	= []
	initLocal	= {count=initcount,counts=initcounts,time=0,tracking=False}
	
//	initIO initialises the monitor process.
	initIO ps
		#	(error,ps)	= openTimer undef timer ps
		|	error<>NoError
			=	abort "monitor could not open timer."
		#	(error,ps)	= openReceiver False (receiver monitorId) ps
		|	error<>NoError
			=	abort "monitor could not open receiver."
		|	otherwise
			=	ps
	
//	window is the single document of the monitor process.
	window	= Window "Monitor" NilLS
				[	WindowId			wId
				,	WindowPos			pos
				,	WindowViewDomain	pDomain
				,	WindowLook			(monitorlook initLocal)
				,	WindowClose			(noLS closeProcess)
				,	WindowInit			[noLS (appPIO (drawInWindow wId [setPenFont font]))]
				]
	
	pDomain	= {corner1=zero,corner2={x=WindowWidth,y=WindowHeight}}
	
//	monitorlook defines the look of the monitor window. 
	monitorlook :: Monitor SelectState UpdateState -> *Picture -> *Picture
	monitorlook {counts,tracking} _ _
			= seq (flatten	[	drawBackground
							,	drawSecondsLine
							,	drawKeyHitsLine
							,	snd (smap drawKeyHitColumn (0,counts))
							:	if tracking [] [drawTotalAndAverage font metrics counts]
							])

//	The timer gathers per second the number of good and bad key hits.
//	The monitor window is updated by drawing only the new diagram bars.
	timer	= Timer ticksPerSecond NilLS
				[	TimerId				tId
				,	TimerSelectState	Unable
				,	TimerFunction		(noLS1 (showKeyHits False))
				]
	
	showKeyHits :: Bool NrOfIntervals (PSt Monitor .p) -> PSt Monitor .p
	showKeyHits final dt monitor=:{ls=local=:{count,counts,time},io}
		#	io		= drawInWindow  wId drawfs io
		#	io		= setWindowLook wId final (monitorlook newlocal) io
		=	{monitor & ls=newlocal,io=io}
	where
		missedcounts= if (dt>1) (repeatn (dt-1) {oks=0,bads=0}) []
		newcounts	= [count:missedcounts]
		newlocal	= {local & count=initcount,counts=counts++newcounts,time=time+dt}
		drawfs		= snd (smap drawKeyHitColumn (time,newcounts))
	
//	The receiver is the interface of the monitor process to the typist process.
	receiver monitorId = Receiver monitorId receive []
	
	receive :: MonitorMessage (Bool,PSt Monitor .p) -> (Bool,PSt Monitor .p)

//	Starting a tracking session enables the timer and clears all previous tracking information.
	receive BeginSession (_,monitor)
		#	local	= {initLocal & tracking=True}
		#	io		= enableTimer  tId monitor.io
		#	io		= setWindowLook wId True (monitorlook local) io
		=	(False,{monitor & ls=local,io=io})

//	For each key hit, only administrate whether it is a good or bad key hit.
	receive (KeyHit char) (_,monitor)
		=	(True,appPLoc (incCount char) monitor)
	where
		incCount :: Char Monitor -> Monitor
		incCount c local=:{count}
			|	c=='\b'
				=	{local & count={count & bads=count.bads+1}}
			|	otherwise
				=	{local & count={count & oks =count.oks +1}}

//	Ending a session disables the timer and presents the number and average of key hits. 
	receive EndSession (firstkeyreceived,monitor=:{ls=local=:{time}})
		#	monitor	= {monitor & ls={local & tracking=False}}
		#	monitor	= showKeyHits True (60-time) monitor
		#	monitor	= appPIO (disableTimer tId) monitor
		=	(firstkeyreceived,monitor)


//	The drawing functions:

drawBackground :: [DrawFunction]
drawBackground
	=	[unfill {corner1=zero,corner2={x=WindowWidth,y=WindowHeight}}]

drawSecondsLine :: [DrawFunction]
drawSecondsLine
	=	[setPenPos {x=GraphX,y=GraphY},draw {vx=GraphWidth,vy=0}:map drawSecond [0,10..MaxNrOfSeconds]]
where
	drawSecond :: Int *Picture -> *Picture
	drawSecond i picture
		#	picture	= drawAt {x=x,y=GraphY} {vx=0,vy=AxisMarkSize}		picture
		#	picture	= drawAt {x=x,y=GraphY+SecondsOffset} (toString i)	picture
		=	picture
	where
		x	= GraphX+i*SecondsWidth

drawKeyHitsLine :: [DrawFunction]
drawKeyHitsLine
	=	[setPenPos {x=GraphX,y=GraphY},draw {vx=0,vy=0-GraphHeight}:map drawKeyHit [0,2..MaxNrKeyHits]]
where
	drawKeyHit :: Int *Picture -> *Picture
	drawKeyHit i picture
		#	picture	= drawAt {x=GraphX-AxisMarkSize,y=y} {vx=AxisMarkSize,vy=0}	picture
		#	picture	= drawAt {x=x,y=y} (toString i)								picture
		=	picture
	where
		x	= GraphX-KeyHitsOffset
		y	= GraphY-i*KeyHitHeight

drawTotalAndAverage :: Font FontMetrics [Count] -> [DrawFunction]
drawTotalAndAverage font metrics counts
	|	isEmpty counts
	=	[]
	=	[totalAndAverage font metrics counts]
where
	totalAndAverage :: Font FontMetrics [Count] *Picture -> *Picture
	totalAndAverage font metrics=:{fMaxWidth} counts picture
		#	(totalW,  picture)	= getFontStringWidth font	totalT				picture
		#	(averageW,picture)	= getFontStringWidth font	averageT			picture
		#	(sumW,    picture)	= getFontStringWidth font	sumT   				picture
		#	picture				= drawAverage	height							picture
		#	picture				= setPenPos		{x=GraphX,y=summaryY}			picture
		#	picture				= draw			totalT							picture
		#	picture				= movePenPos	{vx=0-totalW,vy=lineHeight}		picture
		#	picture				= draw			averageT						picture
		#	picture				= setPenPos		{x=GraphX+(max totalW averageW)+fMaxWidth,y=summaryY}
																				picture
		#	picture				= draw			sumT							picture
		#	picture				= movePenPos	{vx=0-sumW,vy=lineHeight}		picture
		#	picture				= draw			(toString (round 1 average))	picture
		=	picture
	where
		lineHeight		= fontLineHeight metrics
		summaryY		= WindowHeight-SummaryMargin
		seconds			= length counts
		total			= foldr (+) 0 (map (\{oks,bads}->oks-bads) counts)
		average			= toReal total/toReal seconds
		height			= toInt (average*toReal KeyHitHeight)
		sumT			= toString total
		averageT		= "Average:"
		totalT			= "Total:"
		
		drawAverage :: Int *Picture -> *Picture
		drawAverage height picture
			|	height<=0
				=	picture
			|	otherwise
				=	appXorPicture (drawLine {x=GraphX,y=y} {x=GraphX+GraphWidth,y=y}) picture
		where
			y	= GraphY-height

drawKeyHitColumn :: (Int,Count) -> (Int,DrawFunction)
drawKeyHitColumn (i,count)
	=	(i+1,drawColumn i count)
where
	drawColumn :: Int Count *Picture -> *Picture
	drawColumn i {oks,bads} picture
		#	picture	= fill			 {corner1={x=leftX,y=yOk-1},corner2={x=rightX,y=GraphY}} picture
		#	picture	= setPenColour	 Red													 picture
		#	picture	= fill			 {corner1={x=leftX,y=yBad}, corner2={x=rightX,y=GraphY}} picture
		#	picture	= drawSeparation {x=leftX,y=yBad} {x=rightX,y=yBad} bads				 picture
		#	picture	= setPenColour	 Black													 picture
		=	picture
	where
		yBad		= GraphY-bads*KeyHitHeight
		yOk			= GraphY-hits*KeyHitHeight
		leftX		= GraphX+i *  SecondsWidth
		rightX		= leftX +     SecondsWidth
		hits		= oks+bads
		
		drawSeparation :: Point Point Int *Picture -> *Picture
		drawSeparation a b badHits picture
			|	badHits==0
				=	picture
			|	otherwise
				=	drawLine a b (setPenColour White picture)


//	Application constants:

KeyHitHeight	:== 10;		KeyHitsOffset :== 20
MaxNrKeyHits	:== 10
SecondsWidth	:== 4;		SecondsOffset :== 15
MaxNrOfSeconds	:== 60
	
SummaryMargin	:== 20
	
TopMargin		:== 20
LeftMargin		:== 60		// note: LeftMargin  >KeyHitsOffset
RightMargin		:== 30
BottomMargin	:== 60		// note: BottomMargin>SecondsOffset+SummaryMargin
	
GraphWidth		:== MaxNrOfSeconds*SecondsWidth
GraphHeight		:== MaxNrKeyHits  *KeyHitHeight
GraphX			:== LeftMargin
GraphY			:== TopMargin +GraphHeight
AxisMarkSize	:== 3
WindowWidth		:== LeftMargin+RightMargin +GraphWidth
WindowHeight	:== TopMargin +BottomMargin+GraphHeight

		
//	General functions:

round :: !Int !Real -> Real
round decimals r
	|	decimals<=0
		=	toReal (toInt r)
	|	otherwise
		=	toReal (toInt (r*p))/p
		with
			p	= toReal (10^decimals)

smap :: ((.s,.x) -> (.s,.y)) !(.s,![.x]) -> (!.s,![.y])
smap f (s,[x:xs])
	#	(s,y ) = f (s,x)
	#	(s,ys) = smap f (s,xs)
	=	(s,[y:ys])
smap _ (s,_)
	=	(s,[])
